<?php
/**
 * Class Studiare_Theme_Importer
 * Modified version to support URL-based imports
 * @since 0.0.2
 * @category StudiareFramework
 * @package NewsCore WP
 * @author Franklin M Gitonga
 * @link http://studiarethemes.com/
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'STUDI_Studiare_Theme_Importer' ) ) {
    class STUDI_Studiare_Theme_Importer {

        public $directory;
        public $main_path;
        public $theme_options_file;
        public $theme_option_name;
        public $selection_data;
        public $widgets;
        public $content_demo;
        public $content_pages;
        public $flag_as_imported = array();
        public $add_admin_menu = true;
        private static $instance;

        public function __construct() {
            self::$instance = $this;
            $this->flag_as_imported = get_option( 'studi_imported_demo' );
            add_filter( 'add_post_metadata', array( $this, 'check_previous_meta' ), 10, 5 );
            add_action( 'studi_importer_ended', array( $this, 'after_wp_importer' ) );
            $this->process_imports();
        }

        public function check_previous_meta( $continue, $post_id, $meta_key, $meta_value, $unique ) {
            $old_value = get_metadata( 'post', $post_id, $meta_key );
            if ( count( $old_value ) == 1 ) {
                if ( $old_value[0] === $meta_value ) {
                    return false;
                } elseif ( $old_value[0] !== $meta_value ) {
                    update_post_meta( $post_id, $meta_key, $meta_value );
                    return false;
                }
            }
        }

        public function after_wp_importer() {
            update_option( 'studi_imported_demo', $this->flag_as_imported );
        }

        /**
         * Download file from URL to temporary directory
         * @param string $url URL of the file to download
         * @return string|bool Path to downloaded file or false on failure
         */
        private function download_file($url) {
            if (!function_exists('download_url')) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
            }

            $tmp_file = download_url($url);
            
            if (is_wp_error($tmp_file)) {
                error_log('Download error: ' . $tmp_file->get_error_message());
                return false;
            }
            
            return $tmp_file;
        }

        public function process_imports() {
            $content    = false;
            $pages      = false;
            $theme_opts = false;
            $widgets    = false;

            if (!empty($this->selection_data['import_all']) && 1 == $this->selection_data['import_all']) {
                $content = $pages = $theme_opts = $widgets = true;
            } else {
                if (!empty($this->selection_data['import_content']) && 1 == $this->selection_data['import_content']) {
                    $content = true;
                }
                if (!empty($this->selection_data['import_pages']) && 1 == $this->selection_data['import_pages']) {
                    $pages = true;
                }
                if (!empty($this->selection_data['import_opts']) && 1 == $this->selection_data['import_opts']) {
                    $theme_opts = true;
                }
                if (!empty($this->selection_data['import_widgets']) && 1 == $this->selection_data['import_widgets']) {
                    $widgets = true;
                }
            }

            // Handle content import
            if ($content && !empty($this->content_demo)) {
                $downloaded_file = $this->download_file($this->content_demo);
                if ($downloaded_file) {
                    $this->set_demo_data($downloaded_file, true);
                    @unlink($downloaded_file); // Clean up
                } else {
                    error_log('Failed to download content demo file: ' . $this->content_demo);
                }
            } elseif ($pages && !empty($this->content_pages)) {
                $downloaded_file = $this->download_file($this->content_pages);
                if ($downloaded_file) {
                    $this->set_demo_data($downloaded_file, false);
                    @unlink($downloaded_file); // Clean up
                } else {
                    error_log('Failed to download content pages file: ' . $this->content_pages);
                }
            } else {
                echo 'Skip content';
            }

            if ($content || $pages) {
                do_action('studi_importer_content_settings', $this->directory, $this->main_path);
            }

            // Handle theme options import
            if ($theme_opts && !empty($this->theme_options_file)) {
                $downloaded_file = $this->download_file($this->theme_options_file);
                if ($downloaded_file) {
                    $this->set_demo_theme_options($downloaded_file);
                    @unlink($downloaded_file); // Clean up
                } else {
                    error_log('Failed to download theme options file: ' . $this->theme_options_file);
                }
            }

            // Handle widgets import
            if ($widgets && !empty($this->widgets)) {
                $downloaded_file = $this->download_file($this->widgets);
                if ($downloaded_file) {
                    $this->process_widget_import_file($downloaded_file);
                    @unlink($downloaded_file); // Clean up
                } else {
                    error_log('Failed to download widgets file: ' . $this->widgets);
                }
            }

            do_action('studi_importer_ended', $this->directory, $this->main_path);
        }

        public function add_widget_to_sidebar($sidebar_slug, $widget_slug, $count_mod, $widget_settings = array()) {
            $sidebars_widgets = get_option('sidebars_widgets');

            if (!isset($sidebars_widgets[$sidebar_slug])) {
                $sidebars_widgets[$sidebar_slug] = array('_multiwidget' => 1);
            }

            $newWidget = get_option('widget_' . $widget_slug);

            if (!is_array($newWidget)) {
                $newWidget = array();
            }

            $count = count($newWidget) + 1 + $count_mod;
            $sidebars_widgets[$sidebar_slug][] = $widget_slug . '-' . $count;

            $newWidget[$count] = $widget_settings;

            update_option('sidebars_widgets', $sidebars_widgets);
            update_option('widget_' . $widget_slug, $newWidget);
        }

        public function set_demo_data($file, $content = true) {
            if (!defined('WP_LOAD_IMPORTERS')) {
                define('WP_LOAD_IMPORTERS', true);
            }

            if (!defined('IMPORT_DEBUG')) {
                define('IMPORT_DEBUG', false);
            }

            require_once ABSPATH . 'wp-admin/includes/import.php';
            $importer_error = false;

            if (!class_exists('WP_Importer')) {
                $class_wp_importer = ABSPATH . 'wp-admin/includes/class-wp-importer.php';
                if (file_exists($class_wp_importer)) {
                    require $class_wp_importer;
                } else {
                    $importer_error = true;
                }
            }

            require_once dirname(__FILE__) . '/compat.php';
            require_once dirname(__FILE__) . '/parsers/class-wxr-parser.php';
            require_once dirname(__FILE__) . '/parsers/class-wxr-parser-simplexml.php';
            require_once dirname(__FILE__) . '/parsers/class-wxr-parser-xml.php';
            require_once dirname(__FILE__) . '/parsers/class-wxr-parser-regex.php';
            require_once dirname(__FILE__) . '/class-wp-import.php';

            if (!class_exists('WP_Import')) {
                $class_wp_import = dirname(__FILE__) . '/wordpress-importer.php';
                if (file_exists($class_wp_import)) {
                    require_once($class_wp_import);
                } else {
                    $importer_error = true;
                }
            }

            if ($importer_error) {
                die("Error on import");
            } else {
                if ($content) {
                    do_action('studi_importer_before_content', $this->directory, $this->main_path);
                } else {
                    do_action('studi_importer_before_pages', $this->directory, $this->main_path);
                }

                if (!is_file($file)) {
                    echo "The XML file containing the dummy content is not available or could not be read.";
                } else {
                    $wp_import = new WP_Import();
                    $wp_import->fetch_attachments = true;
                    $wp_import->import($file);
                    $this->flag_as_imported[strval($this->directory)]['content'] = true;
                }
            }

            do_action('studi_importer_after_content', $this->directory, $this->main_path);
        }

        public function set_demo_theme_options($file) {
            if (file_exists($file)) {
                do_action('studi_importer_before_theme_options', $this->directory, $this->main_path);

                WP_Filesystem();
                global $wp_filesystem;
                $data = $wp_filesystem->get_contents($file);
                $data = json_decode($data, true);
                $data = maybe_unserialize($data);
                
                if (!empty($data) || is_array($data)) {
                    update_option($this->theme_option_name, $data);
                    $this->flag_as_imported[strval($this->directory)]['options'] = true;
                }

                do_action('studi_importer_after_theme_options', $this->directory, $this->main_path);
            }
        }

        function available_widgets() {
            global $wp_registered_widget_controls;

            $widget_controls = $wp_registered_widget_controls;
            $available_widgets = array();

            foreach ($widget_controls as $widget) {
                if (!empty($widget['id_base']) && !isset($available_widgets[$widget['id_base']])) {
                    $available_widgets[$widget['id_base']]['id_base'] = $widget['id_base'];
                    $available_widgets[$widget['id_base']]['name'] = $widget['name'];
                }
            }

            return apply_filters('studi_importer_widget_available_widgets', $available_widgets);
        }

        function process_widget_import_file($file) {
            if (!file_exists($file)) {
                wp_die(
                    esc_html__('Widget Import file could not be found. Please try again.', 'studiare-importer'),
                    '',
                    array('back_link' => true)
                );
            }

            do_action('studi_importer_before_widgets', $this->directory, $this->main_path);

            WP_Filesystem();
            global $wp_filesystem;
            $data = $wp_filesystem->get_contents($file);
            $data = json_decode($data);

            $this->widget_import_results = $this->import_widgets($data);
        }

        public function import_widgets($data) {
            global $wp_registered_sidebars;

            if (empty($data) || !is_object($data)) {
                return;
            }

            $data = apply_filters('studi_importer_widget_data', $data);
            $available_widgets = $this->available_widgets();
            $widget_instances = array();
            
            foreach ($available_widgets as $widget_data) {
                $widget_instances[$widget_data['id_base']] = get_option('widget_' . $widget_data['id_base']);
            }

            $results = array();

            foreach ($data as $sidebar_id => $widgets) {
                if ('wp_inactive_widgets' == $sidebar_id) {
                    continue;
                }

                if (isset($wp_registered_sidebars[$sidebar_id])) {
                    $sidebar_available = true;
                    $use_sidebar_id = $sidebar_id;
                    $sidebar_message_type = 'success';
                    $sidebar_message = '';
                } else {
                    $sidebar_available = false;
                    $use_sidebar_id = 'wp_inactive_widgets';
                    $sidebar_message_type = 'error';
                    $sidebar_message = esc_html__('Sidebar does not exist in theme (using Inactive)', 'studiare-importer');
                }

                $results[$sidebar_id]['name'] = !empty($wp_registered_sidebars[$sidebar_id]['name']) ? 
                    $wp_registered_sidebars[$sidebar_id]['name'] : $sidebar_id;
                $results[$sidebar_id]['message_type'] = $sidebar_message_type;
                $results[$sidebar_id]['message'] = $sidebar_message;
                $results[$sidebar_id]['widgets'] = array();

                foreach ($widgets as $widget_instance_id => $widget) {
                    $fail = false;
                    $id_base = preg_replace('/-[0-9]+$/', '', $widget_instance_id);
                    $instance_id_number = str_replace($id_base . '-', '', $widget_instance_id);

                    if (!$fail && !isset($available_widgets[$id_base])) {
                        $fail = true;
                        $widget_message_type = 'error';
                        $widget_message = esc_html__('Site does not support widget', 'studiare-importer');
                    }

                    $widget = apply_filters('studi_importer_widget_settings', $widget);

                    if (!$fail && isset($widget_instances[$id_base])) {
                        $sidebars_widgets = get_option('sidebars_widgets');
                        $sidebar_widgets = isset($sidebars_widgets[$use_sidebar_id]) ? 
                            $sidebars_widgets[$use_sidebar_id] : array();

                        $single_widget_instances = !empty($widget_instances[$id_base]) ? 
                            $widget_instances[$id_base] : array();
                            
                        foreach ($single_widget_instances as $check_id => $check_widget) {
                            if (in_array("$id_base-$check_id", $sidebar_widgets) && (array) $widget == $check_widget) {
                                $fail = true;
                                $widget_message_type = 'warning';
                                $widget_message = esc_html__('Widget already exists', 'studiare-importer');
                                break;
                            }
                        }
                    }

                    if (!$fail) {
                        $single_widget_instances = get_option('widget_' . $id_base);
                        $single_widget_instances = !empty($single_widget_instances) ? 
                            $single_widget_instances : array('_multiwidget' => 1);
                        $single_widget_instances[] = (array) $widget;

                        end($single_widget_instances);
                        $new_instance_id_number = key($single_widget_instances);

                        if ('0' === strval($new_instance_id_number)) {
                            $new_instance_id_number = 1;
                            $single_widget_instances[$new_instance_id_number] = $single_widget_instances[0];
                            unset($single_widget_instances[0]);
                        }

                        if (isset($single_widget_instances['_multiwidget'])) {
                            $multiwidget = $single_widget_instances['_multiwidget'];
                            unset($single_widget_instances['_multiwidget']);
                            $single_widget_instances['_multiwidget'] = $multiwidget;
                        }

                        update_option('widget_' . $id_base, $single_widget_instances);

                        $sidebars_widgets = get_option('sidebars_widgets');
                        $new_instance_id = $id_base . '-' . $new_instance_id_number;
                        $sidebars_widgets[$use_sidebar_id][] = $new_instance_id;
                        update_option('sidebars_widgets', $sidebars_widgets);

                        if ($sidebar_available) {
                            $widget_message_type = 'success';
                            $widget_message = esc_html__('Imported', 'studiare-importer');
                        } else {
                            $widget_message_type = 'warning';
                            $widget_message = esc_html__('Imported to Inactive', 'studiare-importer');
                        }
                    }

                    $results[$sidebar_id]['widgets'][$widget_instance_id]['name'] = 
                        isset($available_widgets[$id_base]['name']) ? 
                        $available_widgets[$id_base]['name'] : $id_base;
                    $results[$sidebar_id]['widgets'][$widget_instance_id]['title'] = 
                        !empty($widget->title) ? $widget->title : esc_html__('No Title', 'studiare-importer');
                    $results[$sidebar_id]['widgets'][$widget_instance_id]['message_type'] = $widget_message_type;
                    $results[$sidebar_id]['widgets'][$widget_instance_id]['message'] = $widget_message;
                }
            }

            $this->flag_as_imported[strval($this->directory)]['widgets'] = true;
            do_action('studi_importer_after_widgets', $this->directory, $this->main_path);

            return $results;
        }
    }
}