<?php
namespace Elementor;

class Sc_Pricing_Table extends Widget_Base {
    public function get_name() {
        return 'pricing-table';
    }

    public function get_title() {
        return __('Pricing Table', 'studiare');
    }

    public function get_icon() {
        return 'sc eicon-price-table';
    }

    public function get_categories() {
        return ['studiare'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Content', 'studiare'),
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'plan_title',
            [
                'label' => __('Plan Title', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Basic Plan', 'studiare'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'plan_subtitle',
            [
                'label' => __('Plan Subtitle', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'plan_price',
            [
                'label' => __('Price', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => '29',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'plan_currency',
            [
                'label' => __('Currency Symbol', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => '$',
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'plan_period',
            [
                'label' => __('Period', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Monthly', 'studiare'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'plan_features',
            [
                'label' => __('Features', 'studiare'),
                'type' => Controls_Manager::REPEATER,
                'fields' => [
                    [
                        'name' => 'feature_text',
                        'label' => __('Feature Text', 'studiare'),
                        'type' => Controls_Manager::TEXT,
                        'default' => __('Feature', 'studiare'),
                        'label_block' => true,
                    ],
                    [
                        'name' => 'feature_available',
                        'label' => __('Feature Available', 'studiare'),
                        'type' => Controls_Manager::SWITCHER,
                        'label_on' => __('Yes', 'studiare'),
                        'label_off' => __('No', 'studiare'),
                        'return_value' => 'yes',
                        'default' => 'yes',
                    ],
                ],
                'default' => [
                    ['feature_text' => 'Feature 1', 'feature_available' => 'yes'],
                    ['feature_text' => 'Feature 2', 'feature_available' => 'yes'],
                    ['feature_text' => 'Feature 3', 'feature_available' => 'no'],
                ],
                'title_field' => '{{{ feature_text }}}',
            ]
        );

        $repeater->add_control(
            'button_text',
            [
                'label' => __('Button Text', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Purchase Now', 'studiare'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'button_link',
            [
                'label' => __('Button Link', 'studiare'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'studiare'),
                'default' => [
                    'url' => '',
                    'is_external' => false,
                    'nofollow' => false,
                ],
            ]
        );

        $repeater->add_control(
            'highlight_plan',
            [
                'label' => __('Highlight Plan', 'studiare'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'studiare'),
                'label_off' => __('No', 'studiare'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $repeater->add_control(
            'highlight_badge_text',
            [
                'label' => __('Highlight Badge Text', 'studiare'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Most Popular', 'studiare'),
                'label_block' => true,
                'condition' => [
                    'highlight_plan' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'pricing_plans',
            [
                'label' => __('Pricing Plans', 'studiare'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'plan_title' => __('Basic Plan', 'studiare'),
                        'plan_subtitle' => __('Starter Plan', 'studiare'),
                        'plan_price' => '29',
                        'plan_currency' => '$',
                        'plan_period' => __('Monthly', 'studiare'),
                        'plan_features' => [
                            ['feature_text' => 'Feature 1', 'feature_available' => 'yes'],
                            ['feature_text' => 'Feature 2', 'feature_available' => 'yes'],
                            ['feature_text' => 'Feature 3', 'feature_available' => 'no'],
                        ],
                        'button_text' => __('Purchase Now', 'studiare'),
                        'button_link' => ['url' => ''],
                    ],
                ],
                'title_field' => '{{{ plan_title }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_layout',
            [
                'label' => __('Layout', 'studiare'),
            ]
        );

        $this->add_control(
            'layout_style',
            [
                'label' => __('Layout Style', 'studiare'),
                'type' => Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => __('Default', 'studiare'),
                    'palden' => __('Palden', 'studiare'),
                    'modern' => __('Modern', 'studiare'),
                ],
            ]
        );

        $this->add_responsive_control(
            'columns',
            [
                'label' => __('Columns', 'studiare'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => '1',
                    '2' => '2',
                    '3' => '3',
                    '4' => '4',
                ],
                'default' => '3',
                'tablet_default' => '2',
                'mobile_default' => '1',
                'selectors' => [
                    '{{WRAPPER}} .pricing-table-wrapper' => 'grid-template-columns: repeat({{SIZE}}, 1fr);',
                ],
            ]
        );

        $this->add_responsive_control(
            'container_padding',
            [
                'label' => __('Container Padding', 'studiare'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-table-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'container_margin',
            [
                'label' => __('Container Margin', 'studiare'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-table-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'plan_padding',
            [
                'label' => __('Plan Padding', 'studiare'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '20',
                    'right' => '20',
                    'bottom' => '20',
                    'left' => '20',
                    'unit' => 'px',
                ],
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_responsive_control(
            'plan_margin',
            [
                'label' => __('Plan Margin', 'studiare'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem'],
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'default' => [
                    'top' => '10',
                    'right' => '10',
                    'bottom' => '10',
                    'left' => '10',
                    'unit' => 'px',
                ],
                'condition' => [
                    'layout_style' => ['palden', 'modern'],
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style',
            [
                'label' => __('Style', 'studiare'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'label' => __('Title Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-title',
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => __('Title Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-title' => 'color: {{VALUE}};',
                ],
                'default' => '#333333',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subtitle_typography',
                'label' => __('Subtitle Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-subtitle',
            ]
        );

        $this->add_control(
            'subtitle_color',
            [
                'label' => __('Subtitle Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-subtitle' => 'color: {{VALUE}};',
                ],
                'default' => '#666666',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typography',
                'label' => __('Price Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-price',
            ]
        );

        $this->add_control(
            'price_color',
            [
                'label' => __('Price Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-price' => 'color: {{VALUE}};',
                ],
                'default' => '#f43f54',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'currency_typography',
                'label' => __('Currency Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-currency',
            ]
        );

        $this->add_control(
            'currency_color',
            [
                'label' => __('Currency Symbol Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-currency' => 'color: {{VALUE}};',
                ],
                'default' => '#f43f54',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'period_typography',
                'label' => __('Period Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-period',
            ]
        );

        $this->add_control(
            'period_color',
            [
                'label' => __('Period Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-period' => 'color: {{VALUE}};',
                ],
                'default' => '#666666',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'features_typography',
                'label' => __('Features Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-features li',
            ]
        );

        $this->add_control(
            'features_color',
            [
                'label' => __('Features Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-features li' => 'color: {{VALUE}};',
                ],
                'default' => '#555555',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'label' => __('Button Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'badge_typography',
                'label' => __('Badge Typography', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-badge',
                'condition' => [
                    'layout_style' => 'modern',
                ],
            ]
        );

        $this->add_control(
            'badge_background_color',
            [
                'label' => __('Badge Background Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-badge' => 'background-color: {{VALUE}};',
                ],
                'default' => '#5336ca',
                'condition' => [
                    'layout_style' => 'modern',
                ],
            ]
        );

        $this->add_control(
            'badge_text_color',
            [
                'label' => __('Badge Text Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-badge' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => 'modern',
                ],
            ]
        );

        $this->add_control(
            'plan_background_color',
            [
                'label' => __('Plan Background Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan' => 'background-color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'plan_border',
                'label' => __('Plan Border', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '1',
                            'right' => '1',
                            'bottom' => '1',
                            'left' => '1',
                            'unit' => 'px',
                        ],
                    ],
                    'color' => [
                        'default' => '#ddd',
                    ],
                ],
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'highlight_border',
                'label' => __('Highlight Plan Border', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan.highlight',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '2',
                            'right' => '2',
                            'bottom' => '2',
                            'left' => '2',
                            'unit' => 'px',
                        ],
                    ],
                    'color' => [
                        'default' => '#5336ca',
                    ],
                ],
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'plan_box_shadow',
                'label' => __('Plan Box Shadow', 'studiare'),
                'selector' => '{{WRAPPER}} .pricing-plan',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_control(
            'button_background_color',
            [
                'label' => __('Button Background Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button' => 'background-color: {{VALUE}};',
                ],
                'default' => '#5336ca',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label' => __('Button Text Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_control(
            'button_hover_background_color',
            [
                'label' => __('Button Hover Background Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button:hover, {{WRAPPER}} .pricing-plan-button:focus' => 'background-color: {{VALUE}};',
                ],
                'default' => '#3b2a8c',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_control(
            'button_hover_text_color',
            [
                'label' => __('Button Hover Text Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button:hover, {{WRAPPER}} .pricing-plan-button:focus' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => ['default', 'modern'],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'palden_deco_background',
                'label' => __('Palden Deco Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan .pricing-deco',
                'condition' => [
                    'layout_style' => 'palden',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'gradient',
                    ],
                    'color' => [
                        'default' => '#4097f9',
                    ],
                    'color_b' => [
                        'default' => '#0af0c7',
                    ],
                    'gradient_angle' => [
                        'default' => [
                            'unit' => 'deg',
                            'size' => 135,
                        ],
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'palden_highlight_deco_background',
                'label' => __('Palden Highlight Deco Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan.highlight .pricing-deco',
                'condition' => [
                    'layout_style' => 'palden',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'gradient',
                    ],
                    'color' => [
                        'default' => '#a93bfe',
                    ],
                    'color_b' => [
                        'default' => '#584efd',
                    ],
                    'gradient_angle' => [
                        'default' => [
                            'unit' => 'deg',
                            'size' => 135,
                        ],
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'palden_button_background',
                'label' => __('Palden Button Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan-button',
                'condition' => [
                    'layout_style' => 'palden',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'gradient',
                    ],
                    'color' => [
                        'default' => '#a93bfe',
                    ],
                    'color_b' => [
                        'default' => '#584efd',
                    ],
                    'gradient_angle' => [
                        'default' => [
                            'unit' => 'deg',
                            'size' => 135,
                        ],
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'palden_button_hover_background',
                'label' => __('Palden Button Hover Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan-button:hover, {{WRAPPER}} .pricing-plan-button:focus',
                'condition' => [
                    'layout_style' => 'palden',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'gradient',
                    ],
                    'color' => [
                        'default' => '#fd7d57',
                    ],
                    'color_b' => [
                        'default' => '#f55d59',
                    ],
                    'gradient_angle' => [
                        'default' => [
                            'unit' => 'deg',
                            'size' => 135,
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'palden_button_text_color',
            [
                'label' => __('Palden Button Text Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => 'palden',
                ],
            ]
        );

        $this->add_control(
            'palden_button_hover_text_color',
            [
                'label' => __('Palden Button Hover Text Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-button:hover, {{WRAPPER}} .pricing-plan-button:focus' => 'color: {{VALUE}};',
                ],
                'default' => '#ffffff',
                'condition' => [
                    'layout_style' => 'palden',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'modern_header_background',
                'label' => __('Modern Header Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan .pricing-header',
                'condition' => [
                    'layout_style' => 'modern',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'classic',
                    ],
                    'color' => [
                        'default' => '#f8f9fa',
                    ],
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'modern_highlight_header_background',
                'label' => __('Modern Highlight Header Background', 'studiare'),
                'types' => ['classic', 'gradient'],
                'selector' => '{{WRAPPER}} .pricing-plan.highlight .pricing-header',
                'condition' => [
                    'layout_style' => 'modern',
                ],
                'fields_options' => [
                    'background' => [
                        'default' => 'gradient',
                    ],
                    'color' => [
                        'default' => '#5336ca',
                    ],
                    'color_b' => [
                        'default' => '#3b2a8c',
                    ],
                    'gradient_angle' => [
                        'default' => [
                            'unit' => 'deg',
                            'size' => 135,
                        ],
                    ],
                ],
            ]
        );

        $this->add_control(
            'feature_icon_size',
            [
                'label' => __('Feature Icon Size', 'studiare'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 3,
                    ],
                    'rem' => [
                        'min' => 0.5,
                        'max' => 3,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 16,
                ],
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-feature i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'feature_icon_spacing',
            [
                'label' => __('Feature Icon Spacing', 'studiare'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                    ],
                    'em' => [
                        'min' => 0,
                        'max' => 2,
                    ],
                    'rem' => [
                        'min' => 0,
                        'max' => 2,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 8,
                ],
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-feature i' => 'margin-right: {{SIZE}}{{UNIT}};',
                    'body.rtl {{WRAPPER}} .pricing-plan-feature i' => 'margin-right: 0; margin-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'feature_icon_color_available',
            [
                'label' => __('Feature Available Icon Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-feature i.fa-check' => 'color: {{VALUE}};',
                ],
                'default' => '#5336ca',
            ]
        );

        $this->add_control(
            'feature_icon_color_unavailable',
            [
                'label' => __('Feature Unavailable Icon Color', 'studiare'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .pricing-plan-feature i.fa-times' => 'color: {{VALUE}};',
                ],
                'default' => '#D8D6E3',
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();
        $layout_style = $settings['layout_style'];
        $widget_id = $this->get_id(); // Get unique widget ID
        ?>
        <style>
            /* Main container styles scoped to unique widget ID */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> {
                display: grid;
                gap: 20px;
                margin-bottom: 20px;
            }

            /* Pricing plan base styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan {
                position: relative;
                display: flex;
                flex-direction: column;
                align-items: stretch;
                text-align: center;
                flex: 0 1 330px;
                <?php if ($layout_style === 'palden') : ?>
                    cursor: default;
                    color: #84697c;
                    background: #fff;
                    border-radius: 20px 20px 10px 10px;
                <?php elseif ($layout_style === 'modern') : ?>
                    border-radius: 12px;
                    overflow: hidden;
                    transition: transform 0.3s ease, box-shadow 0.3s ease;
                    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
                <?php endif; ?>
            }

            /* Highlight plan styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan.highlight {
                <?php if ($layout_style === 'default') : ?>
                    transform: scale(1.05);
                <?php elseif ($layout_style === 'palden') : ?>
                    margin: 0;
                    z-index: 10;
                <?php elseif ($layout_style === 'modern') : ?>
                    transform: translateY(-10px);
                    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
                <?php endif; ?>
            }

            /* Hover effects */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan:hover {
                <?php if ($layout_style === 'default') : ?>
                    transform: scale(1.03);
                <?php elseif ($layout_style === 'palden') : ?>
                    .deco-layer--1 {
                        transform: translate3d(15px, 0, 0);
                    }
                    .deco-layer--2 {
                        transform: translate3d(-15px, 0, 0);
                    }
                <?php elseif ($layout_style === 'modern') : ?>
                    transform: translateY(-5px);
                    box-shadow: 0 6px 18px rgba(0, 0, 0, 0.12);
                <?php endif; ?>
            }

            /* Header styles for modern layout */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-header {
                <?php if ($layout_style === 'modern') : ?>
                    padding: 2em 1.5em;
                    position: relative;
                    border-radius: 10px;
                <?php endif; ?>
            }

            /* Highlight badge for modern layout */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan.highlight .pricing-badge {
                <?php if ($layout_style === 'modern') : ?>
                    position: absolute;
                    top: 10px;
                    right: 10px;
                    background: #5336ca;
                    color: #ffffff;
                    padding: 5px 10px;
                    border-radius: 12px;
                    font-size: 0.75em;
                    text-transform: uppercase;
                    font-weight: 600;
                    body.rtl & {
                        right: auto;
                        left: 10px;
                    }
                <?php endif; ?>
            }

            /* Title styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-title {
                font-size: 1.5em;
                margin-bottom: 10px;
                font-weight: 600;
                <?php if ($layout_style === 'palden') : ?>
                    font-size: .75em;
                    text-transform: uppercase;
                <?php elseif ($layout_style === 'modern') : ?>
                    margin: 0;
                    font-size: 1.75em;
                <?php endif; ?>
            }

            /* Subtitle styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-subtitle {
                font-size: 0.9em;
                margin-top: 5px;
                font-weight: 400;
                <?php if ($layout_style === 'modern') : ?>
                    font-size: 0.85em;
                    opacity: 0.8;
                <?php endif; ?>
            }

            /* Price styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-price {
                font-size: 2em;
                font-weight: 700;
                margin: 10px 0;
                <?php if ($layout_style === 'palden') : ?>
                    font-size: 5em;
                    padding: 0;
                    margin: 0 0 .25em 0;
                    line-height: .75;
                <?php elseif ($layout_style === 'modern') : ?>
                    font-size: 2.5em;
                    margin: 0.5em 0;
                <?php endif; ?>
            }

            /* Currency styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-currency {
                <?php if ($layout_style === 'palden') : ?>
                    font-size: .15em;
                    vertical-align: top;
                <?php elseif ($layout_style === 'modern') : ?>
                    font-size: 0.6em;
                    vertical-align: super;
                <?php endif; ?>
            }

            /* Period styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-period {
                font-size: .9em;
                <?php if ($layout_style === 'palden') : ?>
                    font-size: .15em;
                    padding: 0 0 0 .5em;
                    font-style: italic;
                    body.rtl & {
                        padding: 0 .5em 0 0;
                    }
                <?php elseif ($layout_style === 'modern') : ?>
                    font-size: 0.8em;
                    font-style: normal;
                    display: block;
                    margin-top: 5px;
                <?php endif; ?>
            }

            /* Features list styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-features {
                list-style: none;
                padding: 0;
                margin: 20px 0;
                flex-grow: 1;
                <?php if ($layout_style === 'palden') : ?>
                    margin: 0;
                    text-align: center;
                    margin-top: -5em;
                    z-index: 2;
                <?php elseif ($layout_style === 'modern') : ?>
                    padding: 1.5em 2em;
                    background: #fafafa;
                    margin: 0;
                <?php endif; ?>
            }

            /* Feature item styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-features li {
                display: flex;
                align-items: center;
                justify-content: center;
                margin-bottom: 10px;
                <?php if ($layout_style === 'palden') : ?>
                    padding: 0;
                    margin-bottom: 10px !important;
                <?php elseif ($layout_style === 'modern') : ?>
                    justify-content: flex-start;
                    margin-bottom: 12px;
                    font-size: 0.9em;
                <?php endif; ?>
            }
            
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-features li:last-child{
                 margin-bottom: 0;
            }

            /* Button styles */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-button {
                display: inline-block;
                text-decoration: none;
                font-weight: 700;
                transition: background 0.3s ease, transform 0.2s ease, color 0.3s ease;
                text-transform: uppercase;
                <?php if ($layout_style === 'default') : ?>
                    padding: 10px 20px;
                    border-radius: 5px;
                <?php elseif ($layout_style === 'palden') : ?>
                    margin: auto 3em 2em;
                    padding: 1em 2em;
                    border-radius: 30px;
                <?php elseif ($layout_style === 'modern') : ?>
                    padding: 12px 32px;
                    border-radius: 64px;
                    margin: 1.5em auto;
                    font-size: 0.85em;
                <?php endif; ?>
            }

            /* Button hover effects */
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-button:hover,
            .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan-button:focus {
                transform: scale(1.05);
                opacity: 0.9;
                <?php if ($layout_style === 'palden') : ?>
                    background-size: 150%;
                <?php endif; ?>
            }

            /* Palden specific styles */
            <?php if ($layout_style === 'palden') : ?>
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-deco {
                    border-radius: 10px 10px 0 0;
                    padding: 4em 0 9em;
                    position: relative;
                }
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-plan.highlight .pricing-deco {
                    padding: 5em 0 8.885em 0;
                }
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .pricing-deco-img {
                    position: absolute;
                    bottom: 0;
                    left: 0;
                    width: 100%;
                    height: 160px;
                }
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> .deco-layer {
                    transition: transform 0.5s;
                }
            <?php endif; ?>

            /* Media queries for responsive columns */
            @media (min-width: 1024px) {
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> {
                    grid-template-columns: repeat(<?php echo esc_attr($settings['columns']); ?>, 1fr);
                    <?php if ($layout_style === 'palden') : ?>
                        .pricing-plan {
                            margin: 1em -0.5em;
                        }
                    <?php elseif ($layout_style === 'modern') : ?>
                        .pricing-plan {
                            margin: 1em 0.5em;
                        }
                    <?php endif; ?>
                }
            }

            @media (min-width: 768px) and (max-width: 1023px) {
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> {
                    grid-template-columns: repeat(<?php echo esc_attr($settings['columns_tablet']); ?>, 1fr);
                }
            }

            @media (max-width: 767px) {
                .pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> {
                    grid-template-columns: repeat(<?php echo esc_attr($settings['columns_mobile']); ?>, 1fr);
                }
            }
        </style>
        <div class="pricing-table-wrapper pricing-table-wrapper-<?php echo esc_attr($widget_id); ?> pricing-<?php echo esc_attr($layout_style); ?>">
            <?php foreach ($settings['pricing_plans'] as $index => $plan) : ?>
                <div class="pricing-plan <?php echo $plan['highlight_plan'] === 'yes' ? 'highlight' : ''; ?>" data-plan-index="<?php echo $index; ?>">
                    <?php if ($layout_style === 'palden') : ?>
                        <div class="pricing-deco">
                            <svg class="pricing-deco-img" enable-background="new 0 0 300 100" height="100px" preserveAspectRatio="none" version="1.1" viewBox="0 0 300 100" width="300px" xml:space="preserve">
                                <path class="deco-layer deco-layer--1" d="M30.913,43.944c0,0,42.911-34.464,87.51-14.191c77.31,35.14,113.304-1.952,146.638-4.729c48.654-4.056,69.94,16.218,69.94,16.218v54.396H30.913V43.944z" fill="#FFFFFF" opacity="0.6"></path>
                                <path class="deco-layer deco-layer--2" d="M-35.667,44.628c0,0,42.91-34.463,87.51-14.191c77.31,35.141,113.304-1.952,146.639-4.729c48.653-4.055,69.939,16.218,69.939,16.218v54.396H-35.667V44.628z" fill="#FFFFFF" opacity="0.6"></path>
                                <path class="deco-layer deco-layer--3" d="M43.415,98.342c0,0,48.283-68.927,109.133-68.927c65.886,0,97.983,67.914,97.983,67.914v3.716H42.401L43.415,98.342z" fill="#FFFFFF" opacity="0.7"></path>
                                <path class="deco-layer deco-layer--4" d="M-34.667,62.998c0,0,56-45.667,120.316-27.839C167.484,57.842,197,41.332,232.286,30.428c53.07-16.399,104.047,36.903,104.047,36.903l1.333,36.667l-372-2.954L-34.667,62.998z" fill="#FFFFFF"></path>
                            </svg>
                            <div class="pricing-plan-price">
                                <span class="pricing-plan-currency"><?php echo esc_html($plan['plan_currency']); ?></span><?php echo esc_html($plan['plan_price']); ?>
                                <span class="pricing-plan-period"><?php echo esc_html($plan['plan_period']); ?></span>
                            </div>
                            <h2 class="pricing-plan-title"><?php echo esc_html($plan['plan_title']); ?></h2>
                        </div>
                    <?php elseif ($layout_style === 'modern') : ?>
                        <div class="pricing-header">
                            <?php if ($plan['highlight_plan'] === 'yes' && !empty($plan['highlight_badge_text'])) : ?>
                                <span class="pricing-badge"><?php echo esc_html($plan['highlight_badge_text']); ?></span>
                            <?php endif; ?>
                            <h2 class="pricing-plan-title"><?php echo esc_html($plan['plan_title']); ?></h2>
                            <?php if (!empty($plan['plan_subtitle'])) : ?>
                                <div class="pricing-plan-subtitle"><?php echo esc_html($plan['plan_subtitle']); ?></div>
                            <?php endif; ?>
                            <div class="pricing-plan-price">
                                <span class="pricing-plan-currency"><?php echo esc_html($plan['plan_currency']); ?></span><?php echo esc_html($plan['plan_price']); ?>
                                <span class="pricing-plan-period"><?php echo esc_html($plan['plan_period']); ?></span>
                            </div>
                        </div>
                    <?php else : ?>
                        <h2 class="pricing-plan-title"><?php echo esc_html($plan['plan_title']); ?></h2>
                        <?php if (!empty($plan['plan_subtitle'])) : ?>
                            <div class="pricing-plan-subtitle"><?php echo esc_html($plan['plan_subtitle']); ?></div>
                        <?php endif; ?>
                        <div class="pricing-plan-price">
                            <span class="pricing-plan-currency"><?php echo esc_html($plan['plan_currency']); ?></span><?php echo esc_html($plan['plan_price']); ?>
                            <span class="pricing-plan-period"><?php echo esc_html($plan['plan_period']); ?></span>
                        </div>
                    <?php endif; ?>
                    <ul class="pricing-plan-features">
                        <?php if (is_array($plan['plan_features']) && !empty($plan['plan_features'])) : ?>
                            <?php foreach ($plan['plan_features'] as $feature) : ?>
                                <li class="pricing-plan-feature">
                                    <?php if ($feature['feature_available'] === 'yes') : ?>
                                        <i class="fal fa-check"></i>
                                    <?php else : ?>
                                        <i class="fal fa-times"></i>
                                    <?php endif; ?>
                                    <?php echo esc_html($feature['feature_text']); ?>
                                </li>
                            <?php endforeach; ?>
                        <?php else : ?>
                            <li class="pricing-plan-feature"><?php esc_html_e('No features available', 'studiare'); ?></li>
                        <?php endif; ?>
                    </ul>
                    <?php if (!empty($plan['button_text']) && !empty($plan['button_link']['url'])) : ?>
                        <a href="<?php echo esc_url($plan['button_link']['url']); ?>" 
                           class="pricing-plan-button"
                           <?php echo $plan['button_link']['is_external'] ? 'target="_blank"' : ''; ?>
                           <?php echo $plan['button_link']['nofollow'] ? 'rel="nofollow"' : ''; ?>>
                            <?php echo esc_html($plan['button_text']); ?>
                        </a>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
    }

    protected function _content_template() {
        ?>
        <#
        var layout_style = settings.layout_style;
        var widget_id = view.model.get('id'); // Get unique widget ID from Elementor
        var columns = settings.columns || 3;
        var columns_tablet = settings.columns_tablet || 2;
        var columns_mobile = settings.columns_mobile || 1;
        #>
        <style>
            /* Main container styles scoped to unique widget ID */
            .pricing-table-wrapper-{{ widget_id }} {
                display: grid;
                gap: 20px;
                margin-bottom: 20px;
            }

            /* Pricing plan base styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan {
                position: relative;
                display: flex;
                flex-direction: column;
                align-items: stretch;
                text-align: center;
                flex: 0 1 330px;
                <# if ( layout_style === 'palden' ) { #>
                    cursor: default;
                    color: #84697c;
                    background: #fff;
                    border-radius: 20px 20px 10px 10px;
                <# } else if ( layout_style === 'modern' ) { #>
                    border-radius: 12px;
                    overflow: hidden;
                    transition: transform 0.3s ease, box-shadow 0.3s ease;
                    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
                <# } #>
            }

            /* Highlight plan styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan.highlight {
                <# if ( layout_style === 'default' ) { #>
                    transform: scale(1.05);
                <# } else if ( layout_style === 'palden' ) { #>
                    margin: 0;
                    z-index: 10;
                <# } else if ( layout_style === 'modern' ) { #>
                    transform: translateY(-10px);
                    box-shadow: 0 8px 24px rgba(0, 0, 0, 0.15);
                <# } #>
            }

            /* Hover effects */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan:hover {
                <# if ( layout_style === 'default' ) { #>
                    transform: scale(1.03);
                <# } else if ( layout_style === 'palden' ) { #>
                    .deco-layer--1 {
                        transform: translate3d(15px, 0, 0);
                    }
                    .deco-layer--2 {
                        transform: translate3d(-15px, 0, 0);
                    }
                <# } else if ( layout_style === 'modern' ) { #>
                    transform: translateY(-5px);
                    box-shadow: 0 6px 18px rgba(0, 0, 0, 0.12);
                <# } #>
            }

            /* Header styles for modern layout */
            .pricing-table-wrapper-{{ widget_id }} .pricing-header {
                <# if ( layout_style === 'modern' ) { #>
                    padding: 2em 1.5em;
                    position: relative;
                <# } #>
            }

            /* Highlight badge for modern layout */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan.highlight .pricing-badge {
                <# if ( layout_style === 'modern' ) { #>
                    position: absolute;
                    top: 10px;
                    right: 10px;
                    background: #5336ca;
                    color: #ffffff;
                    padding: 5px 10px;
                    border-radius: 12px;
                    font-size: 0.75em;
                    text-transform: uppercase;
                    font-weight: 600;
                    body.rtl & {
                        right: auto;
                        left: 10px;
                    }
                <# } #>
            }

            /* Title styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-title {
                font-size: 1.5em;
                margin-bottom: 10px;
                font-weight: 600;
                <# if ( layout_style === 'palden' ) { #>
                    font-size: .75em;
                    text-transform: uppercase;
                <# } else if ( layout_style === 'modern' ) { #>
                    margin: 0;
                    font-size: 1.75em;
                <# } #>
            }

            /* Subtitle styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-subtitle {
                font-size: 0.9em;
                margin-top: 5px;
                font-weight: 400;
                <# if ( layout_style === 'modern' ) { #>
                    font-size: 0.85em;
                    opacity: 0.8;
                <# } #>
            }

            /* Price styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-price {
                font-size: 2em;
                font-weight: 700;
                margin: 10px 0;
                <# if ( layout_style === 'palden' ) { #>
                    font-size: 5em;
                    padding: 0;
                    margin: 0 0 .25em 0;
                    line-height: .75;
                <# } else if ( layout_style === 'modern' ) { #>
                    font-size: 2.5em;
                    margin: 0.5em 0;
                <# } #>
            }

            /* Currency styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-currency {
                <# if ( layout_style === 'palden' ) { #>
                    font-size: .15em;
                    vertical-align: top;
                <# } else if ( layout_style === 'modern' ) { #>
                    font-size: 0.6em;
                    vertical-align: super;
                <# } #>
            }

            /* Period styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-period {
                font-size: .9em;
                <# if ( layout_style === 'palden' ) { #>
                    font-size: .15em;
                    padding: 0 0 0 .5em;
                    font-style: italic;
                    body.rtl & {
                        padding: 0 .5em 0 0;
                    }
                <# } else if ( layout_style === 'modern' ) { #>
                    font-size: 0.8em;
                    font-style: normal;
                    display: block;
                    margin-top: 5px;
                <# } #>
            }

            /* Features list styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-features {
                list-style: none;
                padding: 0;
                margin: 20px 0;
                flex-grow: 1;
                <# if ( layout_style === 'palden' ) { #>
                    padding: .25em 0 2.5em;
                    text-align: center;
                <# } else if ( layout_style === 'modern' ) { #>
                    padding: 1.5em 2em;
                    background: #fafafa;
                    margin: 0;
                <# } #>
            }

            /* Feature item styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-features li {
                display: flex;
                align-items: center;
                justify-content: center;
                margin-bottom: 10px;
                <# if ( layout_style === 'palden' ) { #>
                    padding: 0;
                <# } else if ( layout_style === 'modern' ) { #>
                    justify-content: flex-start;
                    margin-bottom: 12px;
                    font-size: 0.9em;
                <# } #>
            }

            /* Button styles */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-button {
                display: inline-block;
                text-decoration: none;
                font-weight: 700;
                transition: background 0.3s ease, transform 0.2s ease, color 0.3s ease;
                text-transform: uppercase;
                <# if ( layout_style === 'default' ) { #>
                    padding: 10px 20px;
                    border-radius: 5px;
                <# } else if ( layout_style === 'palden' ) { #>
                    margin: auto 3em 2em;
                    padding: 1em 2em;
                    border-radius: 30px;
                <# } else if ( layout_style === 'modern' ) { #>
                    padding: 12px 32px;
                    border-radius: 64px;
                    margin: 1.5em auto;
                    font-size: 0.85em;
                <# } #>
            }

            /* Button hover effects */
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-button:hover,
            .pricing-table-wrapper-{{ widget_id }} .pricing-plan-button:focus {
                transform: scale(1.05);
                opacity: 0.9;
                <# if ( layout_style === 'palden' ) { #>
                    background-size: 150%;
                <# } #>
            }

            /* Palden specific styles */
            <# if ( layout_style === 'palden' ) { #>
                .pricing-table-wrapper-{{ widget_id }} .pricing-deco {
                    border-radius: 10px 10px 0 0;
                    padding: 4em 0 9em;
                    position: relative;
                }
                .pricing-table-wrapper-{{ widget_id }} .pricing-plan.highlight .pricing-deco {
                    padding: 5em 0 8.885em 0;
                }
                .pricing-table-wrapper-{{ widget_id }} .pricing-deco-img {
                    position: absolute;
                    bottom: 0;
                    left: 0;
                    width: 100%;
                    height: 160px;
                }
                .pricing-table-wrapper-{{ widget_id }} .deco-layer {
                    transition: transform 0.5s;
                }
            <# } #>

            /* Media queries for responsive columns */
            @media (min-width: 1024px) {
                .pricing-table-wrapper-{{ widget_id }} {
                    grid-template-columns: repeat({{ columns }}, 1fr);
                    <# if ( layout_style === 'palden' ) { #>
                        .pricing-plan {
                            margin: 1em -0.5em;
                        }
                    <# } else if ( layout_style === 'modern' ) { #>
                        .pricing-plan {
                            margin: 1em 0.5em;
                        }
                    <# } #>
                }
            }

            @media (min-width: 768px) and (max-width: 1023px) {
                .pricing-table-wrapper-{{ widget_id }} {
                    grid-template-columns: repeat({{ columns_tablet }}, 1fr);
                }
            }

            @media (max-width: 767px) {
                .pricing-table-wrapper-{{ widget_id }} {
                    grid-template-columns: repeat({{ columns_mobile }}, 1fr);
                }
            }
        </style>
        <div class="pricing-table-wrapper pricing-table-wrapper-{{ widget_id }} pricing-{{ layout_style }}">
            <# _.each( settings.pricing_plans, function( plan, index ) { #>
                <div class="pricing-plan <# if ( plan.highlight_plan === 'yes' ) { #>highlight<# } #>" data-plan-index="{{ index }}">
                    <# if ( layout_style === 'palden' ) { #>
                        <div class="pricing-deco">
                            <svg class="pricing-deco-img" enable-background="new 0 0 300 100" height="100px" preserveAspectRatio="none" version="1.1" viewBox="0 0 300 100" width="300px" xml:space="preserve">
                                <path class="deco-layer deco-layer--1" d="M30.913,43.944c0,0,42.911-34.464,87.51-14.191c77.31,35.14,113.304-1.952,146.638-4.729c48.654-4.056,69.94,16.218,69.94,16.218v54.396H30.913V43.944z" fill="#FFFFFF" opacity="0.6"></path>
                                <path class="deco-layer deco-layer--2" d="M-35.667,44.628c0,0,42.91-34.463,87.51-14.191c77.31,35.141,113.304-1.952,146.639-4.729c48.653-4.055,69.939,16.218,69.939,16.218v54.396H-35.667V44.628z" fill="#FFFFFF" opacity="0.6"></path>
                                <path class="deco-layer deco-layer--3" d="M43.415,98.342c0,0,48.283-68.927,109.133-68.927c65.886,0,97.983,67.914,97.983,67.914v3.716H42.401L43.415,98.342z" fill="#FFFFFF" opacity="0.7"></path>
                                <path class="deco-layer deco-layer--4" d="M-34.667,62.998c0,0,56-45.667,120.316-27.839C167.484,57.842,197,41.332,232.286,30.428c53.07-16.399,104.047,36.903,104.047,36.903l1.333,36.667l-372-2.954L-34.667,62.998z" fill="#FFFFFF"></path>
                            </svg>
                            <div class="pricing-plan-price">
                                <span class="pricing-plan-currency">{{{ plan.plan_currency }}}</span>{{{ plan.plan_price }}}
                                <span class="pricing-plan-period">{{{ plan.plan_period }}}</span>
                            </div>
                            <h2 class="pricing-plan-title">{{{ plan.plan_title }}}</h2>
                        </div>
                    <# } else if ( layout_style === 'modern' ) { #>
                        <div class="pricing-header">
                            <# if ( plan.highlight_plan === 'yes' && plan.highlight_badge_text ) { #>
                                <span class="pricing-badge">{{{ plan.highlight_badge_text }}}</span>
                            <# } #>
                            <h2 class="pricing-plan-title">{{{ plan.plan_title }}}</h2>
                            <# if ( plan.plan_subtitle ) { #>
                                <div class="pricing-plan-subtitle">{{{ plan.plan_subtitle }}}</div>
                            <# } #>
                            <div class="pricing-plan-price">
                                <span class="pricing-plan-currency">{{{ plan.plan_currency }}}</span>{{{ plan.plan_price }}}
                                <span class="pricing-plan-period">{{{ plan.plan_period }}}</span>
                            </div>
                        </div>
                    <# } else { #>
                        <h2 class="pricing-plan-title">{{{ plan.plan_title }}}</h2>
                        <# if ( plan.plan_subtitle ) { #>
                            <div class="pricing-plan-subtitle">{{{ plan.plan_subtitle }}}</div>
                        <# } #>
                        <div class="pricing-plan-price">
                            <span class="pricing-plan-currency">{{{ plan.plan_currency }}}</span>{{{ plan.plan_price }}}
                            <span class="pricing-plan-period">{{{ plan.plan_period }}}</span>
                        </div>
                    <# } #>
                    <ul class="pricing-plan-features">
                        <# if ( plan.plan_features && Array.isArray( plan.plan_features ) && plan.plan_features.length > 0 ) { #>
                            <# _.each( plan.plan_features, function( feature ) { #>
                                <li class="pricing-plan-feature">
                                    <# if ( feature.feature_available === 'yes' ) { #>
                                        <i class="fal fa-check"></i>
                                    <# } else { #>
                                        <i class="fal fa-times"></i>
                                    <# } #>
                                    {{{ feature.feature_text }}}
                                </li>
                            <# } ); #>
                        <# } else { #>
                            <li class="pricing-plan-feature"><?php esc_html_e( 'No features available', 'studiare' ); ?></li>
                        <# } #>
                    </ul>
                    <# if ( plan.button_text && plan.button_link.url ) { #>
                        <a href="{{ plan.button_link.url }}"
                           class="pricing-plan-button"
                           <# if ( plan.button_link.is_external ) { #>target="_blank"<# } #>
                           <# if ( plan.button_link.nofollow ) { #>rel="nofollow"<# } #>>
                            {{{ plan.button_text }}}
                        </a>
                    <# } #>
                </div>
            <# } ); #>
        </div>
        <?php
    }
}
?>