<?php
namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Sc_DarkMode extends Widget_Base {

    public function get_name() {
        return 'sc-darkmode';
    }

    public function get_title() {
        return __('Dark Mode Toggle', 'studiare');
    }

    public function get_icon() {
        return 'sc eicon-adjust';
    }

    public function get_categories() {
        return ['studiare'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_content',
            [
                'label' => __('Dark Mode Settings', 'studiare'),
            ]
        );

        $this->add_control(
            'darkmode_note',
            [
                'label' => __('Note', 'studiare'),
                'type' => \Elementor\Controls_Manager::RAW_HTML,
                'raw' => __('This widget will only work if Dark Mode is enabled in the theme settings.', 'studiare'),
                'content_classes' => 'elementor-descriptor',
            ]
        );

        // Icon style selection
        $this->add_control(
            'icon_style',
            [
                'label' => __('Icon Style', 'studiare'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'svg',
                'options' => [
                    'svg' => __('SVG Icon', 'studiare'),
                    'font' => __('Font Icon (e.g., FontAwesome)', 'studiare'),
                ],
            ]
        );

        // Controls for SVG icons
        $this->add_control(
            'svg_icon_light',
            [
                'label' => __('SVG Code for Light Mode', 'studiare'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 3v1m0 16v1m-9-9h1m16 0h1m-2.636-6.364l-.707-.707M6.343 17.657l-.707-.707m12.728 0l-.707.707M6.343 6.343l-.707-.707M12 5a7 7 0 100 14 7 7 0 000-14z"></path></svg>',
                'condition' => [
                    'icon_style' => 'svg',
                ],
            ]
        );

        $this->add_control(
            'svg_icon_dark',
            [
                'label' => __('SVG Code for Dark Mode', 'studiare'),
                'type' => \Elementor\Controls_Manager::TEXTAREA,
                'default' => '<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 12.79A9 9 0 1111.21 3a7 7 0 009.79 9.79z"></path></svg>',
                'condition' => [
                    'icon_style' => 'svg',
                ],
            ]
        );

        // Controls for font icons
        $this->add_control(
            'font_icon_light',
            [
                'label' => __('Font Icon for Light Mode', 'studiare'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fal fa-sun',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_style' => 'font',
                ],
            ]
        );

        $this->add_control(
            'font_icon_dark',
            [
                'label' => __('Font Icon for Dark Mode', 'studiare'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fal fa-moon',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'icon_style' => 'font',
                ],
            ]
        );

        $this->end_controls_section();

        // Style controls for the button
        $this->start_controls_section(
            'section_style',
            [
                'label' => __('Style', 'studiare'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        // Responsive button size
        $this->add_responsive_control(
            'button_size',
            [
                'label' => __('Button Size', 'studiare'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'vw', 'vh', 'rem', 'em'],
                'range' => [
                    'px' => [
                        'min' => 24,
                        'max' => 100,
                        'step' => 1,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                    'em' => [
                        'min' => 1,
                        'max' => 10,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 46,
                ],
                'selectors' => [
                    '{{WRAPPER}} .top-bar-darkmode' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        // Icon size
        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __('Icon Size', 'studiare'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'rem', 'em'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 0.5,
                        'max' => 5,
                    ],
                    'em' => [
                        'min' => 0.5,
                        'max' => 5,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 24,
                ],
                'selectors' => [
                    '{{WRAPPER}} .darkmode-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .darkmode-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        // Tabs for light and dark mode styles
        $this->start_controls_tabs('style_tabs');

        // Light mode styles
        $this->start_controls_tab(
            'style_light_tab',
            [
                'label' => __('Light Mode', 'studiare'),
            ]
        );

        $this->add_control(
            'icon_color_light',
            [
                'label' => __('Icon Color (Light Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#6c8d9d',
                'selectors' => [
                    '{{WRAPPER}} .top-bar-darkmode .dark-mode-toggle' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_light',
            [
                'label' => __('Button Background (Light Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#f5f6f5',
                'selectors' => [
                    '{{WRAPPER}} .top-bar-darkmode' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover_light',
            [
                'label' => __('Button Background Hover (Light Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#e8ecef',
                'selectors' => [
                    '{{WRAPPER}} .top-bar-darkmode:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        // Dark mode styles
        $this->start_controls_tab(
            'style_dark_tab',
            [
                'label' => __('Dark Mode', 'studiare'),
            ]
        );

        $this->add_control(
            'icon_color_dark',
            [
                'label' => __('Icon Color (Dark Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    'body.scdarkcolors {{WRAPPER}} .top-bar-darkmode .dark-mode-toggle' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_dark',
            [
                'label' => __('Button Background (Dark Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#2e2e2e',
                'selectors' => [
                    'body.scdarkcolors {{WRAPPER}} .top-bar-darkmode' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover_dark',
            [
                'label' => __('Button Background Hover (Dark Mode)', 'studiare'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#3e3e3e',
                'selectors' => [
                    'body.scdarkcolors {{WRAPPER}} .top-bar-darkmode:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        if (!function_exists('codebean_option')) {
            return;
        }

        // Check if dark mode is enabled and the button should be shown in the header
        $show_darkmode_icon = codebean_option('show_darkmode_icon_in_header') ?? false;
        $dark_mode_ready = codebean_option('sc_darkmode_ready') ?? false;

        if (!$dark_mode_ready || !$show_darkmode_icon) {
            return;
        }

        ?>
        <div class="top-bar-darkmode">
            <a href="#" class="dark-mode-toggle">
                <span class="darkmode-icon">
                    <span class="dark-mode-icon-light">
                        <?php if ($settings['icon_style'] === 'svg') : ?>
                            <?php echo $settings['svg_icon_light']; ?>
                        <?php else : ?>
                            <?php \Elementor\Icons_Manager::render_icon($settings['font_icon_light'], ['aria-hidden' => 'true']); ?>
                        <?php endif; ?>
                    </span>
                    <span class="dark-mode-icon-dark" style="display: none;">
                        <?php if ($settings['icon_style'] === 'svg') : ?>
                            <?php echo $settings['svg_icon_dark']; ?>
                        <?php else : ?>
                            <?php \Elementor\Icons_Manager::render_icon($settings['font_icon_dark'], ['aria-hidden' => 'true']); ?>
                        <?php endif; ?>
                    </span>
                </span>
            </a>
        </div>
        <?php
    }

    protected function content_template() {
        ?>
        <div class="top-bar-darkmode">
            <a href="#" class="dark-mode-toggle">
                <span class="darkmode-icon">
                    <span class="dark-mode-icon-light">
                        <# if ( settings.icon_style === 'svg' ) { #>
                            {{{ settings.svg_icon_light }}}
                        <# } else { #>
                            <i class="{{ settings.font_icon_light.value }}" aria-hidden="true"></i>
                        <# } #>
                    </span>
                    <span class="dark-mode-icon-dark" style="display: none;">
                        <# if ( settings.icon_style === 'svg' ) { #>
                            {{{ settings.svg_icon_dark }}}
                        <# } else { #>
                            <i class="{{ settings.font_icon_dark.value }}" aria-hidden="true"></i>
                        <# } #>
                    </span>
                </span>
            </a>
        </div>
        <?php
    }
}